---
title: Introduction to Workflows · Cloudflare Learning Paths
description: >
  Cloudflare Workflows provides durable execution capabilities, allowing
  developers to create reliable, repeatable workflows that run in the
  background. Workflows are designed to resume execution even if the underlying
  compute fails, ensuring that tasks complete eventually. They are built on top
  of Cloudflare Workers and handle scaling and provisioning automatically.
lastUpdated: 2025-08-20T18:47:44.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/learning-paths/workflows-course/series/workflows-1/
  md: https://developers.cloudflare.com/learning-paths/workflows-course/series/workflows-1/index.md
---

* Watch this episode

  Cloudflare Workflows provides durable execution capabilities, allowing developers to create reliable, repeatable workflows that run in the background. Workflows are designed to resume execution even if the underlying compute fails, ensuring that tasks complete eventually. They are built on top of Cloudflare Workers and handle scaling and provisioning automatically.

  Workflows are triggered by events, such as Event Notifications consumed from a Queue, HTTP requests, another Worker, or even scheduled timers. Individual steps within a Workflow are designed as retriable units of work. The state is persisted between steps, allowing workflows to resume from the last successful step after failures. Workflows automatically generate metrics for each step, aiding in debugging and observability.

  **Related content**

  If you want to dive into detail, refer to the following pages:

  * [Source code for the Punderful repository](https://github.com/craigsdennis/punderful-workflows)
  * [Cloudflare Workflows](https://developers.cloudflare.com/workflows/)
  * [Cloudflare Workers AI](https://developers.cloudflare.com/workers-ai/)

* Step-by-step tutorial

  Punderful is a sample application that showcases the use of various Cloudflare primitives, including Workers, D1, Vectorize, Workers AI, and Workflows. The application displays a list of puns stored in a D1 database.

  The homepage lists the latest puns stored in D1. The application also includes a semantic search feature powered by Vectorize. To perform a search:

  1. Go to the Punderful search page.
  2. Type a search query in the "Search for a pun..." input box.
  3. Observe the search results appearing instantly below the search box.

  To demonstrate adding a new pun:

  1. Go to the Punderful creation page.
  2. Enter a new pun in the "Enter your pun here..." textarea.
  3. Observe the preview of the pun updating as you type.
  4. Click the "Submit Pun" button.

  When a new pun is submitted, it needs to be indexed in Vectorize for the semantic search to work. This indexing process involves creating embeddings from the pun text. This is a task suitable for background processing using Cloudflare Workflows, avoiding delays for the user in the request-response loop.

  ### Implementing a Workflow to Process New Puns

  A workflow is implemented to handle the background processing required when a new pun is submitted.

  #### Triggering the Workflow

  When a new pun is submitted via the `/api/puns` endpoint, the data is first inserted into the D1 database. Then, a new Workflow instance is created and triggered to perform the subsequent background tasks.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/index.tsx#L165)

  In this handler, `c.env.PUBLISH.create(crypto.randomUUID(), { punId, pun: payload.pun })` creates a new instance of the workflow bound as `PUBLISH`, assigns it a unique ID, and passes the `punId` and `pun` text as the payload.

  #### Defining the Workflow Class

  The workflow logic is defined in a class that extends `WorkflowEntrypoint`.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L12)

  The `run` method is the entrypoint for the workflow execution. It receives the `event` containing the payload and a `step` object to define individual, durable steps.

  #### Workflow Steps

  Each discrete, retriable task in the workflow is defined using `await step.do()`.

  ##### Content Moderation

  Optionally, the workflow can perform content moderation using an external service like OpenAI's moderation API if an API key is available in the environment.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L16)

  This step calls the OpenAI moderation API. If the content is flagged as inappropriate, the pun's status is updated in the database, and a `NonRetryableError` is thrown. Throwing a `NonRetryableError` prevents the workflow from retrying this step, as the content is permanently deemed inappropriate.

  ##### Creating Embeddings

  Next, create vector embeddings for the pun text using a Workers AI model.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L34)

  This step uses the `@cf/baai/bge-large-en-v1.5` model from Workers AI to generate a vector embedding for the `pun` text. The result (the embedding vector) is returned by the step and can be used in subsequent steps. `step.do()` ensures this step will be retried if it fails, guaranteeing that embeddings are eventually created.

  ##### Categorizing the Pun

  Optionally, use a Workers AI language model to categorize the pun.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L41)

  This step uses the `@cf/meta/llama-3.1-8b-instruct` model with a specific system prompt to generate categories for the pun. The generated categories string is returned by the step. This step also benefits from `step.do()`'s reliability.

  ##### Adding Embeddings to Vectorize

  Insert the created pun embedding and potentially categories embedding into the Vectorize database.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L78)

  This step uses `this.env.VECTORIZE.upsert()` to add the generated embeddings and associated metadata to the Vectorize database. This makes the pun searchable semantically. `step.do()` ensures this critical indexing step is completed reliably.

  ##### Updating Database Status

  The final step updates the status of the pun in the D1 database to indicate that it has been published and processed by the workflow.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/7cec7f4bd7d6b17085cb6d6cb3e56b6a4b5b7c9d/src/workflows/publish.ts#L104)

  This step updates the `status` column in the D1 database to "published" for the corresponding pun ID. Once this step is complete, the pun is considered fully processed and ready to be displayed on the homepage.

  #### Workflow Bindings

  To make the `PublishWorkflow` class available to the main Worker and to provide access to necessary resources (like D1, AI, Vectorize), bindings are configured in the `wrangler.toml` file.

  [See here](https://github.com/craigsdennis/punderful-workflows/blob/main/wrangler.toml)

  This configuration defines a workflow named `publish`, binds it to the environment variable `PUBLISH`, and links it to the `PublishWorkflow` class in `src/index.ts`. It also shows bindings for Workers AI (`AI`) and Vectorize (`VECTORIZE`), which are accessed via `this.env` within the workflow.

  ### Vectorize for Semantic Search

  Vectorize is a vector database used in this application to enable semantic search for puns. It stores the vector embeddings created by Workers AI. The search functionality queries this Vectorize index to find puns similar in meaning to the user's query.

  The homepage displays recently published puns (status "published"). The detail page for a specific pun displays "Similar Puns", which are found by querying Vectorize with the embedding of the current pun.

  ### Scalability

  Cloudflare Workers and Workflows are designed to scale automatically based on demand, handling concurrent requests and background tasks efficiently without requiring manual provisioning.

* Series overview

  [Watch Episode 1: Understand Cloudflare Workflows ](https://developers.cloudflare.com/learning-paths/workflows-course/series/workflows-1/)In this episode, we introduce Cloudflare Workflows, which provides durable execution capabilities, allowing developers to create reliable, repeatable workflows that run in the background.

  [Watch Episode 2: Monitor and batch your website data ](https://developers.cloudflare.com/learning-paths/workflows-course/series/workflows-2/)In this episode, we describe how Workflows can be used to process batches of data, ensuring each item in the batch goes through a defined process with reliable execution.

  [Watch Episode 3: Use cron triggers to develop time-aware applications ](https://developers.cloudflare.com/learning-paths/workflows-course/series/workflows-3/)In this episode, we review Workflows ability to explicitly schedule tasks using cron triggers and pause execution with \`step.sleep\` allows developers to build sophisticated, time-aware applications.
